/*

Modelspoor stack modules
www.stackmodules.nl
Ed den Ouden 2020-2024



Library:      DCC_Decoder
Bestand:      BB DCC-decoder - SB ontkoppelen
Functie:      switch voor 16 elektromagnetische ontkoppelaars, drukknop (DCC-adres 1-2048)
Board:        Nano V3 328p
Kanalen:      elke ontkoppelaar gebruikt 1 DCC-kanaal
Opmerking:    deze Arduino library heeft een actieve DCC-locomotief op het spoor nodig om correct te werken
Configuratie: zie variabelen bij 'Door de gebruiker in te stellen variabelen'



Adressering:  deze DCC-decoder heeft 16 kanalen/uitgangen en de nummering begint bij DCC-adres 1 (of 17, 33, 49, 65, etc (groepen van 16))

              Stel dat het DCCStartadres 17 is, de adressering van het eerste kanaal wordt dan in Rocrail (adres-poort): 5 1 en in iTrain 17

               DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain
              adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres
              --------------------   --------------------   --------------------   --------------------   --------------------
                1    1   1    1        17   5   1    17       33   9   1    33       49   13  1    49       65   17  1    65
                2    1   2    2        18   5   2    18       34   9   2    34       50   13  2    50       66   17  2    66
                3    1   3    3        19   5   3    19       35   9   3    35       51   13  3    51       67   17  3    67
                4    1   4    4        20   5   4    20       36   9   4    36       52   13  4    52
                5    2   1    5        21   6   1    21       37   10  1    37       53   14  1    43       Et cetera
                6    2   2    6        22   6   2    22       38   10  2    38
                7    2   3    7        23   6   3    23       39   10  3    39       Et cetera...
                8    2   4    8        24   6   4    24       40   10  4    40
                9    3   1    9        25   7   1    25       41   11  1    41
                10   3   2    10       26   7   2    26       42   11  2    42
                11   3   3    11       27   7   3    27       43   11  3    43
                12   3   4    12       28   7   4    28       44   11  4    44
                13   4   1    13       29   8   1    29       45   12  1    45
                14   4   2    14       30   8   2    30       46   12  2    46
                15   4   3    16       31   8   3    31       47   12  3    47
                16   4   4    16       32   8   4    32       48   12  4    48

*/



// Door de gebruiker in te stellen variabelen

// #define SERIALMON  // Comment/uncomment schakelt de seriële monitor uit/aan, default = uit

const int dccStartAddress = 17;     // Eerste DCC-adres van deze 16 kanaals decoder (16 ontkoppelaars, adressen 1-2048), de andere 15 kanalen nummeren oplopend door
const boolean outputIdle = LOW;     // Output indien kanaal niet actief
const boolean outputActive = HIGH;  // Output indien kanaal wel actief

// Pinmapping voor basis board DCC-decoder
// Kanalen         1  2  3  4  5  6  7  8   9   10  11  12  13  14  15  16
int usedPins[] = { 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 14, 15, 16, 17, 18, 19 };

// Einde door de gebruiker in te stellen variabelen



#include <DCC_Decoder.h>
//#define DEBUGGER

const byte numAccessories = 16;  // Aantal kanalen van deze decoder

// Interne datastructuur
typedef struct DCCAccessoryData {
  int address;     // DCC-adres
  byte outputpin;  // Arduino pin
  byte dccstate;   // 1=on, 0=off
};
DCCAccessoryData accessory[numAccessories];



void BasicAccDecoderPacket_Handler(int address, boolean activate, byte data) {

  address -= 1;
  address *= 4;
  address += 1;
  address += (data & 0x06) >> 1;
  //address = address - 4; // Uncomment voor Roco Maus of z21, tenzij je deze instelling al in de MultiMaus zelf hebt aangezet



#ifdef SERIALMON
  Serial.print("DCC-adres, activeer, data: ");
  Serial.print(address);
  Serial.print(", ");
  Serial.print(activate);
  Serial.print(", ");
  Serial.println(data);
#endif



  boolean enable = (data & 0x01) ? 1 : 0;

  for (byte i = 0; i < numAccessories; i++) {
    if (address == accessory[i].address) {
      if (enable) accessory[i].dccstate = 1;
      else accessory[i].dccstate = 0;
    }
  }



#ifdef DEBUGGER
  Serial.print(address);
  Serial.print(" - ");
  Serial.print(enable);
  if (enable == 1) Serial.println(" groen/aan");
  if (enable == 0) Serial.println(" rood/uit");
#endif
}



void setup() {

#ifdef SERIALMON
  Serial.begin(115200);
  Serial.println("Modelspoor Stack Module V2.1 (Nano V3 328p) - Ed den Ouden - 2024");
  Serial.println("DCC-decoder voor 16 ontkoppelaars (www.stackmodules.nl)");
  Serial.println("");
  Serial.print("Eerste DCC-adres: ");
  Serial.println(dccStartAddress);
  Serial.print("Aantal kanalen: ");
  Serial.println(numAccessories);
  Serial.print("Outputs inactief: ");
  Serial.println(outputIdle);
  Serial.print("Outputs actief: ");
  Serial.println(outputActive);
  Serial.println("");
  Serial.println("Initialisatie klaar, wacht op DCC-instructies van de centrale...");
  Serial.println("");
#endif



  // Toewijzing pinnen en DCC-adressen (niet wijzigen)

  accessory[0].address = dccStartAddress;
  accessory[0].outputpin = usedPins[0];

  accessory[1].address = dccStartAddress + 1;
  accessory[1].outputpin = usedPins[1];

  accessory[2].address = dccStartAddress + 2;
  accessory[2].outputpin = usedPins[2];

  accessory[3].address = dccStartAddress + 3;
  accessory[3].outputpin = usedPins[3];

  accessory[4].address = dccStartAddress + 4;
  accessory[4].outputpin = usedPins[4];

  accessory[5].address = dccStartAddress + 5;
  accessory[5].outputpin = usedPins[5];

  accessory[6].address = dccStartAddress + 6;
  accessory[6].outputpin = usedPins[6];

  accessory[7].address = dccStartAddress + 7;
  accessory[7].outputpin = usedPins[7];

  accessory[8].address = dccStartAddress + 8;
  accessory[8].outputpin = usedPins[8];

  accessory[9].address = dccStartAddress + 9;
  accessory[9].outputpin = usedPins[9];

  accessory[10].address = dccStartAddress + 10;
  accessory[10].outputpin = usedPins[10];

  accessory[11].address = dccStartAddress + 11;
  accessory[11].outputpin = usedPins[11];

  accessory[12].address = dccStartAddress + 12;
  accessory[12].outputpin = usedPins[12];

  accessory[13].address = dccStartAddress + 13;
  accessory[13].outputpin = usedPins[13];

  accessory[14].address = dccStartAddress + 14;
  accessory[14].outputpin = usedPins[14];

  accessory[15].address = dccStartAddress + 15;
  accessory[15].outputpin = usedPins[15];



  DCC.SetBasicAccessoryDecoderPacketHandler(BasicAccDecoderPacket_Handler, true);
  DCC.SetupDecoder(0x00, 0x00, 0);



  for (byte i = 0; i < numAccessories; i++) {
    pinMode(accessory[i].outputpin, OUTPUT);
    digitalWrite(accessory[i].outputpin, outputIdle);
  }
}



void loop() {

  static int n = 0;
  DCC.loop();
  if (++n >= numAccessories) n = 0;

  if (accessory[n].dccstate)
    digitalWrite(accessory[n].outputpin, outputActive);
  else
    digitalWrite(accessory[n].outputpin, outputIdle);
}
