/*

Modelspoor stack modules
www.stackmodules.nl
Ed den Ouden 2020-2024



Library:      DCC_Decoder
Bestand:      BB DCC-decoder - SB wissels
Functie:      switch voor 8 elektromagnetische wissels (instelbare pulse of aan/uit, DCC-adres 1-2048)
Board:        Nano V3 328p
Kanalen:      elke wissel gebruikt 2 DCC-kanalen: adres = rechtdoor en adres + 1 = afbuigen
Opmerking:    deze Arduino library heeft een actieve DCC-locomotief op het spoor nodig om correct te werken
Configuratie: zie variabelen bij 'Door de gebruiker in te stellen variabelen #1' en 'Door de gebruiker in te stellen variabelen #2'



Adressering:  deze DCC-decoder heeft 16 uitgangen voor 8 wissels, en de nummering begint bij DCC-adres 1 (of 17, 33, 49 etc (groepen van 16))

              Stel dat het DCCStartadres 17 is, de adressering van het eerste kanaal wordt dan in Rocrail (adres-poort): 5 1 en in iTrain 17

               DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain    DCC |Rocrail|iTrain
              adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres
              --------------------   --------------------   --------------------   --------------------   --------------------
                1    1   1    1        17   5   1    17       33   9   1    33       49   13  1    49       65   17  1    65
                2    1   2    2        18   5   2    18       34   9   2    34       50   13  2    50       66   17  2    66
                3    1   3    3        19   5   3    19       35   9   3    35       51   13  3    51       67   17  3    67
                4    1   4    4        20   5   4    20       36   9   4    36       52   13  4    52
                5    2   1    5        21   6   1    21       37   10  1    37       53   14  1    43       Et cetera
                6    2   2    6        22   6   2    22       38   10  2    38
                7    2   3    7        23   6   3    23       39   10  3    39       Et cetera...
                8    2   4    8        24   6   4    24       40   10  4    40
                9    3   1    9        25   7   1    25       41   11  1    41
                10   3   2    10       26   7   2    26       42   11  2    42
                11   3   3    11       27   7   3    27       43   11  3    43
                12   3   4    12       28   7   4    28       44   11  4    44
                13   4   1    13       29   8   1    29       45   12  1    45
                14   4   2    14       30   8   2    30       46   12  2    46
                15   4   3    16       31   8   3    31       47   12  3    47
                16   4   4    16       32   8   4    32       48   12  4    48

              Rocrail skipt bij wissels steeds 1 adres (te weten 2 4 6 8 10 12 14 16, etc), dat adres wordt automatisch gebruikt als 2e kanaal van het wissel
*/



// Door de gebruiker in te stellen variabelen #1

// #define SERIALMON  // Comment/uncomment schakelt de seriële monitor uit/aan, default = uit

const int dccStartAddress = 17;     // Eerste DCC-adres van deze 16 kanaals decoder (8 wissels, adressen 1-2048), de andere 15 kanalen nummeren oplopend door
const int pulseTime = 250;          // Pulsetijd in ms voor de wissels, 0 is schakel on/off zonder pulse
const boolean outputIdle = LOW;     // Output indien kanaal niet actief
const boolean outputActive = HIGH;  // Output indien kanaal wel actief

// Pinmapping voor basis board DCC-decoder
// Kanalen         1  2  3  4  5  6  7  8   9   10  11  12  13  14  15  16
int usedPins[] = { 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 14, 15, 16, 17, 18, 19 };

// Einde door de gebruiker in te stellen variabelen #1



#include <DCC_Decoder.h>
#define kDCC_INTERRUPT 0

const byte numAccessories = 16;

// Interne datastructuur
typedef struct {
  int address;             // DCC-adres
  byte output;             // Status output 1=on (aan), 0=off (uit)
  int outputPin;           // Arduino pin
  boolean isDigital;       // True = digital, false=analog. Indien analoog, definieer dan ook de analogValue variabele
  boolean isFlasher;       // True = flash output, false=no time, no flash
  byte analogValue;        // Waarde voor type analoog (PWM 0-255)
  int durationMilli;       // Milliseconden output aan, 0=constant aan
  unsigned long onMilli;   // Interne variabele
  unsigned long offMilli;  // Interne variabele
} DCCAccessoryAddress;

DCCAccessoryAddress gAddresses[numAccessories];



// Door de gebruiker in te stellen variabelen #2

void ConfigureDecoder() {

  gAddresses[0].address = dccStartAddress;  // Deze parameter niet wijzigen
  gAddresses[0].output = 1;
  gAddresses[0].outputPin = usedPins[0];    // Deze parameter niet wijzigen
  gAddresses[0].isDigital = true;
  gAddresses[0].isFlasher = false;
  gAddresses[0].analogValue = 0;
  gAddresses[0].durationMilli = pulseTime;

  gAddresses[1].address = dccStartAddress + 1;
  gAddresses[1].output = 1;
  gAddresses[1].outputPin = usedPins[1];
  gAddresses[1].isDigital = true;
  gAddresses[1].isFlasher = false;
  gAddresses[1].analogValue = 0;
  gAddresses[1].durationMilli = pulseTime;

  gAddresses[2].address = dccStartAddress + 2;
  gAddresses[2].output = 1;
  gAddresses[2].outputPin = usedPins[2];
  gAddresses[2].isDigital = true;
  gAddresses[2].isFlasher = false;
  gAddresses[2].analogValue = 0;
  gAddresses[2].durationMilli = pulseTime;

  gAddresses[3].address = dccStartAddress + 3;
  gAddresses[3].output = 1;
  gAddresses[3].outputPin = usedPins[3];
  gAddresses[3].isDigital = true;
  gAddresses[3].isFlasher = false;
  gAddresses[3].analogValue = 0;
  gAddresses[3].durationMilli = pulseTime;

  gAddresses[4].address = dccStartAddress + 4;
  gAddresses[4].output = 1;
  gAddresses[4].outputPin = usedPins[4];
  gAddresses[4].isDigital = true;
  gAddresses[4].isFlasher = false;
  gAddresses[4].analogValue = 0;
  gAddresses[4].durationMilli = pulseTime;

  gAddresses[5].address = dccStartAddress + 5;
  gAddresses[5].output = 1;
  gAddresses[5].outputPin = usedPins[5];
  gAddresses[5].isDigital = true;
  gAddresses[5].isFlasher = false;
  gAddresses[5].analogValue = 0;
  gAddresses[5].durationMilli = pulseTime;

  gAddresses[6].address = dccStartAddress + 6;
  gAddresses[6].output = 1;
  gAddresses[6].outputPin = usedPins[6];
  gAddresses[6].isDigital = true;
  gAddresses[6].isFlasher = false;
  gAddresses[6].analogValue = 0;
  gAddresses[6].durationMilli = pulseTime;

  gAddresses[7].address = dccStartAddress + 7;
  gAddresses[7].output = 1;
  gAddresses[7].outputPin = usedPins[7];
  gAddresses[7].isDigital = true;
  gAddresses[7].isFlasher = false;
  gAddresses[7].analogValue = 0;
  gAddresses[7].durationMilli = pulseTime;

  gAddresses[8].address = dccStartAddress + 8;
  gAddresses[8].output = 1;
  gAddresses[8].outputPin = usedPins[8];
  gAddresses[8].isDigital = true;
  gAddresses[8].isFlasher = false;
  gAddresses[8].analogValue = 0;
  gAddresses[8].durationMilli = pulseTime;

  gAddresses[9].address = dccStartAddress + 9;
  gAddresses[9].output = 1;
  gAddresses[9].outputPin = usedPins[9];
  gAddresses[9].isDigital = true;
  gAddresses[9].isFlasher = false;
  gAddresses[9].analogValue = 0;
  gAddresses[9].durationMilli = pulseTime;

  gAddresses[10].address = dccStartAddress + 10;
  gAddresses[10].output = 1;
  gAddresses[10].outputPin = usedPins[10];
  gAddresses[10].isDigital = true;
  gAddresses[10].isFlasher = false;
  gAddresses[10].analogValue = 0;
  gAddresses[10].durationMilli = pulseTime;

  gAddresses[11].address = dccStartAddress + 11;
  gAddresses[11].output = 1;
  gAddresses[11].outputPin = usedPins[11];
  gAddresses[11].isDigital = true;
  gAddresses[11].isFlasher = false;
  gAddresses[11].analogValue = 0;
  gAddresses[11].durationMilli = pulseTime;

  gAddresses[12].address = dccStartAddress + 12;
  gAddresses[12].output = 1;
  gAddresses[12].outputPin = usedPins[12];
  gAddresses[12].isDigital = true;
  gAddresses[12].isFlasher = false;
  gAddresses[12].analogValue = 0;
  gAddresses[12].durationMilli = pulseTime;

  gAddresses[13].address = dccStartAddress + 13;
  gAddresses[13].output = 1;
  gAddresses[13].outputPin = usedPins[13];
  gAddresses[13].isDigital = true;
  gAddresses[13].isFlasher = false;
  gAddresses[13].analogValue = 0;
  gAddresses[13].durationMilli = pulseTime;

  gAddresses[14].address = dccStartAddress + 14;
  gAddresses[14].output = 1;
  gAddresses[14].outputPin = usedPins[14];
  gAddresses[14].isDigital = true;
  gAddresses[14].isFlasher = false;
  gAddresses[14].analogValue = 0;
  gAddresses[14].durationMilli = pulseTime;

  gAddresses[15].address = dccStartAddress + 15;
  gAddresses[15].output = 1;
  gAddresses[15].outputPin = usedPins[15];
  gAddresses[15].isDigital = true;
  gAddresses[15].isFlasher = false;
  gAddresses[15].analogValue = 0;
  gAddresses[15].durationMilli = pulseTime;

  // Einde door de gebruiker in te stellen variabelen #2



  for (int i = 0; i < numAccessories; i++) {
    if (gAddresses[i].outputPin) {
      pinMode(gAddresses[i].outputPin, OUTPUT);
    }
    gAddresses[i].onMilli = 0;
    gAddresses[i].offMilli = 0;
  }
}



void BasicAccDecoderPacket_Handler(int address, boolean activate, byte data) {

  address -= 1;
  address *= 4;
  address += 1;
  address += (data & 0x06) >> 1;
  address = address - 4; // Uncomment deze regel voor Roco MultiMaus of z21, tenzij je dit reeds in de Multi Maus zelf hebt ingesteld

  boolean enable = (data & 0x01) ? 1 : 0;

  for (int i = 0; i < numAccessories; i++) {

    if (address == gAddresses[i].address)  // Controleert of dit DCC-adres bij deze decoder hoort
    {

#ifdef SERIALMON
      Serial.print("DCC packet (address, activate, data): ");
      Serial.print(address, DEC);
      Serial.print(", ");
      Serial.print(activate);
      Serial.print(", ");
      Serial.println(data);
#endif

      if (enable)  // Deze routine controleert de DCC-instructie: 1/enable/rood/turnoff of 0/disable/groen/straight, en bereidt de outputs voor
      {
        gAddresses[i + 1].output = 1;  // 1 enable rood
        gAddresses[i + 1].onMilli = millis();
        gAddresses[i + 1].offMilli = 0;

      } else {
        gAddresses[i].output = 1;  // 0 disable groen
        gAddresses[i].onMilli = millis();
        gAddresses[i].offMilli = 0;
      }
    }
  }
}



void setup() {

#ifdef SERIALMON
  Serial.begin(115200);
  Serial.println("Modelspoor Stack Module V2.1 (Nano V3 328p) - Ed den Ouden - 2024");
  Serial.println("DCC-decoder voor 8 elektromagnetische wissels (www.stackmodules.nl)");
  Serial.println("");
  Serial.print("Eerste DCC-adres: ");
  Serial.println(dccStartAddress);
  Serial.print("Aantal kanalen: ");
  Serial.println(numAccessories);
  Serial.print("Outputs inactief: ");
  Serial.println(outputIdle);
  Serial.print("Outputs actief: ");
  Serial.println(outputActive);
  Serial.print("Pulsetijd in ms: ");
  Serial.println(pulseTime);
  Serial.println("");
  Serial.println("Initialisatie klaar, wacht op DCC-instructies van de centrale...");
  Serial.println("");
#endif

  DCC.SetBasicAccessoryDecoderPacketHandler(BasicAccDecoderPacket_Handler, true);
  ConfigureDecoder();
  DCC.SetupDecoder(0x00, 0x00, kDCC_INTERRUPT);
}



void loop() {

  static int addr = 0;
  DCC.loop();

  // Verhoog naar het volgende adres/kanaal
  if (++addr >= numAccessories) {
    addr = 0;
  }



  // Output 0/uit/groen/rechtdoor
  if (gAddresses[addr].offMilli && gAddresses[addr].offMilli < millis()) {

    gAddresses[addr].offMilli = 0;

    if (gAddresses[addr].isDigital) {
      digitalWrite(gAddresses[addr].outputPin, outputIdle);
    } else {
      analogWrite(gAddresses[addr].outputPin, 0);
    }

    // Indien enabled en type flash, zet aan tijd
    if (gAddresses[addr].output && gAddresses[addr].isFlasher) {
      gAddresses[addr].onMilli = millis() + gAddresses[addr].durationMilli;
    } else {
      gAddresses[addr].output = 0;
    }
    return;
  }



  // Output 1/aan/rood/afbuigen
  if (gAddresses[addr].onMilli && gAddresses[addr].onMilli <= millis()) {

    gAddresses[addr].onMilli = 0;

    if (gAddresses[addr].isDigital) {
      digitalWrite(gAddresses[addr].outputPin, outputActive);
    } else {
      analogWrite(gAddresses[addr].outputPin, gAddresses[addr].analogValue);
    }

    // Indien enabled en type flash, zet uit tijd
    if (gAddresses[addr].durationMilli) {
      gAddresses[addr].offMilli = millis() + gAddresses[addr].durationMilli;
    }
    return;
  }
}
