/*

Modelspoor stack modules
www.stackmodules.nl
Ed den Ouden 2020-2024



Library:      LocoNet
Bestand:      BB LocoNet-T LM311 client - Outputs ontkoppelen
Functie:      switch voor 16 ontkoppelaars (aan/uit, druktoets)
Board:        Nano V3 328p
Kanalen:      16 
Configuratie: zie variabelen bij 'Door de gebruiker in te stellen variabelen'
Bugfix:       20-12-2023, afwijking address -1



Adressering:  deze LocoNet client heeft 16 kanalen/uitgangen en de nummering begint bij LocoNet adres 1 (of 17, 33, 49, 65, etc (meestal groepen van 16))

              Stel dat het loconetStartadres 17 is, de adressering van het eerste kanaal wordt dan in Rocrail (adres-poort): 5 1 en in iTrain 17

              LocoN|Rocrail|iTrain   LocoN|Rocrail|iTrain   LocoN|Rocrail|iTrain   LocoN|Rocrail|iTrain   LocoN|Rocrail|iTrain
              adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres    adres|adr prt|adres
              --------------------   --------------------   --------------------   --------------------   --------------------
                1    1   1    1        17   5   1    17       33   9   1    33       49   13  1    49       65   17  1    65
                2    1   2    2        18   5   2    18       34   9   2    34       50   13  2    50       66   17  2    66
                3    1   3    3        19   5   3    19       35   9   3    35       51   13  3    51       67   17  3    67
                4    1   4    4        20   5   4    20       36   9   4    36       52   13  4    52
                5    2   1    5        21   6   1    21       37   10  1    37       53   14  1    43       Et cetera
                6    2   2    6        22   6   2    22       38   10  2    38
                7    2   3    7        23   6   3    23       39   10  3    39       Et cetera...
                8    2   4    8        24   6   4    24       40   10  4    40
                9    3   1    9        25   7   1    25       41   11  1    41
                10   3   2    10       26   7   2    26       42   11  2    42
                11   3   3    11       27   7   3    27       43   11  3    43
                12   3   4    12       28   7   4    28       44   11  4    44
                13   4   1    13       29   8   1    29       45   12  1    45
                14   4   2    14       30   8   2    30       46   12  2    46
                15   4   3    16       31   8   3    31       47   12  3    47
                16   4   4    16       32   8   4    32       48   12  4    48
*/



// Door de gebruiker in te stellen variabelen

// #define SERIALMON  // Comment/uncomment schakelt de seriële monitor uit/aan (default = uit)

int loconetStartadres = 49  // LocoNet startadres van deze stack module (1-4096), de volgende 15 kanalen nummeren oplopend
int aantalKanalen = 16;     // Aantal kanalen/pinnen van deze stack module (default = 16)

// Pinmapping basis board LocoNet, pinnen 6 en 8 worden door LocoNet gebruikt
// Kanalen:        1  2  3  4  5  6  7   8   9   10  11  12  13  14  15  16
int usedPins[] = { 2, 3, 4, 5, 7, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 };

// Einde door de gebruiker in te stellen variabelen



#include <LocoNet.h>  // Library
lnMsg *LnPacket;      // Pointer operator



void setup() {

  LocoNet.init();  // Initialiseer Loconet library

  // Definieer 16 pinnen van de pinmapping als output
  for (byte i = 0; i < aantalKanalen; i++) {
    pinMode(usedPins[i], OUTPUT);
  }

#ifdef SERIALMON
  Serial.begin(57600);
  Serial.println("Modelspoor Stack Module V2.1 (Nano V3 328p) - Ed den Ouden - 2024");
  Serial.println("LocoNet outputs voor ontkoppelaars (www.stackmodules.nl)");
  Serial.println("");
  Serial.print("LocoNet startadres: ");
  Serial.println(loconetStartadres);
  Serial.print("Aantal kanalen: ");
  Serial.println(aantalKanalen);
  Serial.println("");
  Serial.println("Initialisatie klaar, wacht op LocoNet-instructies van de centrale...");
  Serial.println();
#endif
}



void loop() {

  // Check for any received LocoNet packets
  LnPacket = LocoNet.receive();

  if (LnPacket) {
// First print out the packet in HEX
#ifdef SERIALMON
    Serial.print("RX: ");
#endif
    uint8_t msgLen = getLnMsgSize(LnPacket);
    for (uint8_t x = 0; x < msgLen; x++) {
      uint8_t val = LnPacket->data[x];

// Print a leading 0 if less than 16 to make 2 HEX digits
#ifdef SERIALMON
      if (val < 16)
        Serial.print('0');

      Serial.print(val, HEX);
      Serial.print(' ');
#endif
    }

    // If this packet was not a Switch or Sensor Message then print a new line
    if (!LocoNet.processSwitchSensorMessage(LnPacket)) {
      Serial.println();
    }
  }
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Sensor messages
void notifySensor(uint16_t Address, uint8_t State) {

#ifdef SERIALMON
  Serial.print("Sensor: ");
  Serial.print(Address, DEC);
  Serial.print(" - ");
  Serial.println(State ? "Active" : "Inactive");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Request messages
void notifySwitchRequest(uint16_t Address, uint8_t Output, uint8_t Direction) {

  // Bepaal of het ontvangen adres bij deze stack module hoort en schakel vervolgens de opdracht (aan of uit)
  if (Address >= loconetStartadres && Address <= (loconetStartadres + aantalKanalen)) {

#ifdef SERIALMON
    Serial.print("Adres: ");
    Serial.print(Address, DEC);
    Serial.print(" - ");
    // Serial.print(Direction ? "Rechtdoor" : "Afbuigen");
    // Serial.print(" - ");
    Serial.println(Output ? "Aan" : "Uit");
    Serial.println();
#endif

    digitalWrite(usedPins[Address - loconetStartadres], Output);  // Schakel het LocoNetadres aan of uit
  }
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Output Report messages
void notifySwitchOutputsReport(uint16_t Address, uint8_t ClosedOutput, uint8_t ThrownOutput) {

#ifdef SERIALMON
  Serial.print("Switch Outputs Report: ");
  Serial.print(Address, DEC);
  Serial.print(": Closed - ");
  Serial.print(ClosedOutput ? "On" : "Off");
  Serial.print(": Thrown - ");
  Serial.println(ThrownOutput ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch Sensor Report messages
void notifySwitchReport(uint16_t Address, uint8_t State, uint8_t Sensor) {

#ifdef SERIALMON
  Serial.print("Switch Sensor Report: ");
  Serial.print(Address, DEC);
  Serial.print(':');
  Serial.print(Sensor ? "Switch" : "Aux");
  Serial.print(" - ");
  Serial.println(State ? "Active" : "Inactive");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Switch State messages
void notifySwitchState(uint16_t Address, uint8_t Output, uint8_t Direction) {

#ifdef SERIALMON
  Serial.print("Switch State: ");
  Serial.print(Address, DEC);
  Serial.print(':');
  Serial.print(Direction ? "Closed" : "Thrown");
  Serial.print(" - ");
  Serial.println(Output ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all Power messages
void notifyPower(uint8_t State) {

#ifdef SERIALMON
  Serial.print("Layout Power State: ");
  Serial.println(State ? "On" : "Off");
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all MultiSensePower messages
void notifyMultiSensePower(uint8_t BoardID, uint8_t Subdistrict, uint8_t Mode, uint8_t Direction) {

#ifdef SERIALMON
  Serial.print("MultiSensePower: Board ID: ");
  Serial.print(BoardID, DEC);
  Serial.print(" Sub District: ");
  Serial.print(Subdistrict, DEC);
  Serial.print(" Mode: ");
  Serial.print(Mode, DEC);
  Serial.print(" Direction: ");
  Serial.println(Direction, DEC);
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all notifyMultiSenseTransponder messages
void notifyMultiSenseTransponder(uint16_t Address, uint8_t Zone, uint16_t LocoAddress, uint8_t Present) {

#ifdef SERIALMON
  Serial.print("MultiSenseTransponder: Address: ");
  Serial.print(Address, DEC);
  Serial.print(" Zone: ");
  Serial.print(Zone, DEC);
  Serial.print(" Loco Address: ");
  Serial.print(LocoAddress, DEC);
  Serial.print(" Present: ");
  Serial.println(Present, DEC);
#endif
}



// This call-back function is called from LocoNet.processSwitchSensorMessage
// for all LongAck messages

void notifyLongAck(uint8_t d1, uint8_t d2) {

#ifdef SERIALMON
  Serial.print("LongACK : Data Byte 1: ");
  Serial.print(d1, DEC);
  Serial.print(" Data Byte 2: ");
  Serial.println(d2, DEC);
#endif
}
