/*

Modelspoor stack modules
www.stackmodules.nl
Ed den Ouden 2020-2024



Library:      LocoNet Github
Bestand:      BB LocoClient-T LM311 - Inputs stroomdetectie
Functie:      15 sensoren stroomdetectie (aan/uit)
Board:        Nano V3 328p
Kanalen:      15x sensoren op 16 kanalen
Configuratie: zie variabelen bij 'Door de gebruiker in te stellen variabelen'



Adressering:  deze LocoClient heeft 15 kanalen/ingangen en de nummering begint bij adres 1 (of 17, 33, 49, 65, etc (groepen van 16))

              Let op: pin 13 (XX) is niet bruikbaar bij LocoClient stroomdetectie (interne LED op de Nano), er zijn dus 15 ingangen!

              Stel dat het startadres 17 is, de adressering wordt dan bv in Rocrail:

              LN  RR   LN  RR   LN  RR   LN  RR   LN  RR   LN  RR
              ------   ------   ------   ------   ------   ------
              1   1    17  17   33  33   49  49   65  65   81  81
              2   2    18  18   34  34   50  50   66  66   Etc...
              3   3    19  19   35  35   51  51   Etc...
              4   4    20  20   36  36   Etc...
              5   5    21  21   37  37
              6   6    22  22   38  38
              7   7    23  23   39  39
              8   8    24  24   40  40
              9   9    25  25   41  41
              XX  XX   XX  XX   XX  XX   <- pin 13 (zie pinmapping) is niet bruikbaar als input met pullup door LED_BUILTIN
              11  11   27  27   43  43
              12  12   28  28   44  44
              13  13   29  29   45  45
              14  14   30  30   46  46
              15  15   31  31   47  47
              16  16   32  32   48  48

              Dit is een zg 'flat' adressering (FADA, 17=17, de meeste terugmelders gebruiken dit zo), en geen combinatie adres/poort (17 = adres 5, poort 2) zoals bv bij wissels of andere accessoires

*/



// Door de gebruiker in te stellen variabelen

//#define SERIALMON  // Comment/uncomment schakelt de seriële monitor uit/aan (default is uit)

const int LoconetStartadres = 81;   // Eerste adres van deze 16 kanaals terugmelder, de volgende 15 kanalen nummeren oplopend
int aantalKanalen = 16;             // Aantal kanalen/pinnen van deze stack module (default is 16)
const boolean outputIdle = LOW;     // Output indien kanaal niet actief (default is LOW)
const boolean outputActive = HIGH;  // Output indien kanaal wel actief (default is HIGH)
const int bounceDelay = 150;        // Korte delays voor debouncing, varieer hiermee indien de melder 'stottert' (default is 150 milliseconden, hoger = minder stotteren, lager is sneller reageren))

// Pinmapping voor basis board LocoClient, pinnen 6 en 8 worden intern gebruikt
// Kanaal:         1  2  3  4  5  6  7   8   9   10  11  12  13  14  15  16     // Let op kanaal 10 (= pin 13)
int usedPins[] = { 2, 3, 4, 5, 7, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 };  // Let op pin 13 (= kanaal 10)

// Einde door de gebruiker in te stellen variabelen



#include <LocoNet.h>  // Library

// Arrays voor de sensoren
byte sensorAan[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
byte sensorWaarde[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };



void setup() {

#ifdef SERIALMON
  Serial.begin(115200);
  Serial.println("Modelspoor Stack Module V2.1 (Nano V3 328p) - Ed den Ouden - 2024");
  Serial.println("LocoClient terugmelding stroomdetectie (www.stackmodules.nl)");
  Serial.println("");
  Serial.print("Startadres: ");
  Serial.println(LoconetStartadres);
  Serial.print("Aantal kanalen: ");
  Serial.println(aantalKanalen);
  Serial.println("");
#endif

  // Definieer 16 pinnen van de pinmapping als inputpinnen en zet corresponderende sensorAan op false (0)
  for (byte i = 0; i < aantalKanalen; i++) {
    if (i == 9) { pinMode(usedPins[i], INPUT); }  // Pin 13 geen pullup door LED_BUILTIN
    pinMode(usedPins[i], INPUT_PULLUP);
  }

  LocoNet.init();  // Initialiseer library

#ifdef SERIALMON
  Serial.println("Initialisatie klaar, wacht op (terug)meldingen van de sensoren...");
  Serial.println();
#endif
}



void loop() {

  // Lees alle pinnen uit (aantalKanalen)
  for (byte i = 0; i < aantalKanalen; i++) {
    sensorWaarde[i] = digitalRead(usedPins[i]);
  }

  // Loop om alle kanalen te lezen/schakelen
  for (byte j = 0; j < aantalKanalen; j++) {

    // Sensor idle, geen detectie op deze sensor
    if (sensorWaarde[j] > 0 && !sensorAan[j]) {
      LocoNet.reportSensor(LoconetStartadres + j, outputIdle);
      sensorAan[j] = 1;
      delay(bounceDelay);

#ifdef SERIALMON
      Serial.print("Sensorwaarde = ");
      Serial.print(sensorWaarde[j]);
      Serial.print(", sensoradres ");
      Serial.print(LoconetStartadres + j);
      Serial.println(" OFF");
#endif
    }

    // Sensor active, melding gedetecteerd op deze sensor
    if (sensorWaarde[j] == 0 && sensorAan[j]) {
      LocoNet.reportSensor(LoconetStartadres + j, outputActive);
      sensorAan[j] = 0;
      delay(bounceDelay);

#ifdef SERIALMON
      Serial.print("Sensorwaarde = ");
      Serial.print(sensorWaarde[j]);
      Serial.print(", sensoradres ");
      Serial.print(LoconetStartadres + j);
      Serial.println(" ON");
#endif
    }
  }
}
